<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Primero verificar si hay sesión
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

require_once 'config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

// Obtener estadísticas
$visitas_hoy = 0;
$visitas_activas = 0;
$total_usuarios = 0;

try {
    // Visitas hoy
    $query = "SELECT COUNT(*) as total FROM visitas WHERE DATE(fecha_registro) = CURDATE()";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $visitas_hoy = $result ? $result['total'] : 0;
    
    // Visitas activas
    $query = "SELECT COUNT(*) as total FROM visitas WHERE hora_salida IS NULL AND DATE(fecha_visita) = CURDATE()";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $visitas_activas = $result ? $result['total'] : 0;
    
    // Total usuarios
    $query = "SELECT COUNT(*) as total FROM usuarios WHERE activo = 1";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $total_usuarios = $result ? $result['total'] : 0;
    
} catch (PDOException $e) {
    error_log("Error en consultas: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8f9fa;
            margin: 0;
            padding: 0;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .main-content {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .card-stat {
            border: none;
            border-radius: 15px;
            transition: transform 0.3s;
            overflow: hidden;
            margin-bottom: 20px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
        }
        .card-stat:hover {
            transform: translateY(-5px);
        }
        .card-stat.visitas {
            background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
            border-left: 5px solid var(--celeste-oscuro);
        }
        .card-stat.activas {
            background: linear-gradient(135deg, #fff3cd 0%, #ffeaa7 100%);
            border-left: 5px solid #ff9800;
        }
        .card-stat.usuarios {
            background: linear-gradient(135deg, #d4edda 0%, #c3e6cb 100%);
            border-left: 5px solid #28a745;
        }
        .icon-stat {
            font-size: 2.5rem;
            opacity: 0.8;
        }
        .welcome-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            margin-bottom: 20px;
            padding: 20px;
            animation: slideIn 0.5s ease-out;
        }
        @keyframes slideIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .navbar-top {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            padding: 10px 20px;
            margin-bottom: 20px;
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link active" href="dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="modules/visitas/registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link" href="modules/visitas/salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link" href="modules/visitas/listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link" href="modules/reportes/generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <?php if (isset($_SESSION['user_tipo']) && $_SESSION['user_tipo'] == 'admin'): ?>
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link" href="modules/usuarios/gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link" href="modules/areas/gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link" href="modules/entidad/configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        <?php endif; ?>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-content">
                <!-- Top Navbar -->
                <nav class="navbar navbar-expand navbar-light navbar-top">
                    <div class="container-fluid">
                        <div class="collapse navbar-collapse">
                            <h1 class="h2 mb-0">Dashboard</h1>
                            <div class="navbar-nav ms-auto">
                                <span class="navbar-text">
                                    <i class="fas fa-user me-1"></i>
                                    <?php echo isset($_SESSION['user_nombre']) ? htmlspecialchars($_SESSION['user_nombre']) : 'Usuario'; ?>
                                    <span class="badge bg-primary ms-2">
                                        <?php echo isset($_SESSION['user_tipo']) ? htmlspecialchars($_SESSION['user_tipo']) : ''; ?>
                                    </span>
                                </span>
                            </div>
                        </div>
                    </div>
                </nav>

                <!-- Welcome Card -->
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="welcome-card">
                            <div class="row align-items-center">
                                <div class="col-md-8">
                                    <h3 class="mb-3">¡Bienvenido, 
                                        <?php 
                                        if (isset($_SESSION['user_nombre'])) {
                                            $nombres = explode(' ', $_SESSION['user_nombre']);
                                            echo htmlspecialchars($nombres[0]);
                                        } else {
                                            echo 'Usuario';
                                        }
                                        ?>!
                                    </h3>
                                    <p class="text-muted">Sistema de registro y control de visitas para la entidad pública.</p>
                                    <div class="d-flex gap-2">
                                        <a href="modules/visitas/registrar.php" class="btn btn-primary">
                                            <i class="fas fa-plus-circle me-2"></i>Nueva Visita
                                        </a>
                                        <a href="modules/visitas/salida.php" class="btn btn-warning">
                                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                                        </a>
                                        <a href="modules/reportes/generar.php" class="btn btn-success">
                                            <i class="fas fa-chart-bar me-2"></i>Ver Reportes
                                        </a>
                                    </div>
                                </div>
                                <div class="col-md-4 text-center">
                                    <i class="fas fa-calendar-check fa-5x text-primary"></i>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Statistics Cards -->
                <div class="row">
                    <div class="col-md-4">
                        <div class="card card-stat visitas p-3">
                            <div class="card-body">
                                <div class="row align-items-center">
                                    <div class="col-8">
                                        <h5 class="card-title">Visitas Hoy</h5>
                                        <h2 class="mb-0"><?php echo $visitas_hoy; ?></h2>
                                        <small>Total de visitas registradas hoy</small>
                                    </div>
                                    <div class="col-4 text-end">
                                        <i class="fas fa-users icon-stat text-primary"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="card card-stat activas p-3">
                            <div class="card-body">
                                <div class="row align-items-center">
                                    <div class="col-8">
                                        <h5 class="card-title">Visitas Activas</h5>
                                        <h2 class="mb-0"><?php echo $visitas_activas; ?></h2>
                                        <small>Visitas sin registrar salida</small>
                                    </div>
                                    <div class="col-4 text-end">
                                        <i class="fas fa-user-clock icon-stat text-warning"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="card card-stat usuarios p-3">
                            <div class="card-body">
                                <div class="row align-items-center">
                                    <div class="col-8">
                                        <h5 class="card-title">Usuarios Activos</h5>
                                        <h2 class="mb-0"><?php echo $total_usuarios; ?></h2>
                                        <small>Usuarios del sistema</small>
                                    </div>
                                    <div class="col-4 text-end">
                                        <i class="fas fa-user-shield icon-stat text-success"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Recent Visits -->
                <div class="row mt-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header bg-white">
                                <h5 class="mb-0">Últimas Visitas Registradas</h5>
                            </div>
                            <div class="card-body">
                                <?php
                                try {
                                    $query = "SELECT v.*, a.nombre_area as area 
                                              FROM visitas v 
                                              LEFT JOIN areas a ON v.area_id = a.id 
                                              WHERE DATE(v.fecha_visita) = CURDATE() 
                                              ORDER BY v.hora_ingreso DESC 
                                              LIMIT 10";
                                    $stmt = $db->prepare($query);
                                    $stmt->execute();
                                    
                                    if ($stmt->rowCount() > 0):
                                ?>
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Hora Ingreso</th>
                                                <th>Visitante</th>
                                                <th>DNI</th>
                                                <th>Área</th>
                                                <th>Motivo</th>
                                                <th>Estado</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php while ($row = $stmt->fetch(PDO::FETCH_ASSOC)): ?>
                                            <tr>
                                                <td><?php echo date('H:i', strtotime($row['hora_ingreso'])); ?></td>
                                                <td><?php echo htmlspecialchars($row['apellidos_visitante'] . ', ' . $row['nombres_visitante']); ?></td>
                                                <td><?php echo htmlspecialchars($row['documento_visitante']); ?></td>
                                                <td><?php echo htmlspecialchars($row['area']); ?></td>
                                                <td><?php echo htmlspecialchars(substr($row['motivo_visita'], 0, 30)) . '...'; ?></td>
                                                <td>
                                                    <?php if ($row['hora_salida']): ?>
                                                        <span class="badge bg-success">Completada</span>
                                                    <?php else: ?>
                                                        <span class="badge bg-warning">En curso</span>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endwhile; ?>
                                        </tbody>
                                    </table>
                                </div>
                                <?php else: ?>
                                <div class="text-center py-4">
                                    <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                    <p class="text-muted">No hay visitas registradas hoy</p>
                                </div>
                                <?php endif; 
                                } catch (PDOException $e) {
                                    echo '<div class="alert alert-danger">Error al cargar visitas: ' . htmlspecialchars($e->getMessage()) . '</div>';
                                }
                                ?>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Script simple para el dashboard
        document.addEventListener('DOMContentLoaded', function() {
            console.log('Dashboard cargado correctamente');
            
            // Actualizar la hora cada minuto
            function actualizarHora() {
                const ahora = new Date();
                document.getElementById('hora-actual').textContent = 
                    ahora.toLocaleTimeString('es-ES', { hour: '2-digit', minute: '2-digit' });
            }
            
            // Inicializar hora
            actualizarHora();
            setInterval(actualizarHora, 60000);
        });
    </script>
</body>
</html>