<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Verificar sesión y permisos de administrador
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

// Verificar que sea administrador
if (!isset($_SESSION['user_tipo']) || $_SESSION['user_tipo'] != 'admin') {
    $_SESSION['mensaje'] = "No tienes permisos para acceder a esta sección";
    $_SESSION['tipo_mensaje'] = "danger";
    header("Location: ../../dashboard.php");
    exit();
}

require_once '../../config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

// Variables
$mensaje = '';
$tipo_mensaje = '';
$area_editar = null;

// Procesar acciones
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (isset($_POST['accion'])) {
        switch ($_POST['accion']) {
            case 'crear':
                $nombre_area = trim($_POST['nombre_area'] ?? '');
                $descripcion = trim($_POST['descripcion'] ?? '');
                $responsable = trim($_POST['responsable'] ?? '');
                $telefono_contacto = trim($_POST['telefono_contacto'] ?? '');
                
                if (empty($nombre_area)) {
                    $mensaje = "El nombre del área es obligatorio";
                    $tipo_mensaje = "danger";
                } else {
                    try {
                        // Verificar si el área ya existe
                        $query = "SELECT id FROM areas WHERE nombre_area = :nombre";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':nombre', $nombre_area);
                        $stmt->execute();
                        
                        if ($stmt->rowCount() > 0) {
                            $mensaje = "Ya existe un área con ese nombre";
                            $tipo_mensaje = "danger";
                        } else {
                            $query = "INSERT INTO areas (nombre_area, descripcion, responsable, telefono_contacto) 
                                      VALUES (:nombre, :descripcion, :responsable, :telefono)";
                            
                            $stmt = $db->prepare($query);
                            $stmt->bindParam(':nombre', $nombre_area);
                            $stmt->bindParam(':descripcion', $descripcion);
                            $stmt->bindParam(':responsable', $responsable);
                            $stmt->bindParam(':telefono', $telefono_contacto);
                            
                            if ($stmt->execute()) {
                                $mensaje = "Área creada exitosamente";
                                $tipo_mensaje = "success";
                                
                                // Limpiar formulario
                                $_POST = array();
                            } else {
                                $mensaje = "Error al crear el área";
                                $tipo_mensaje = "danger";
                            }
                        }
                    } catch (PDOException $e) {
                        $mensaje = "Error: " . $e->getMessage();
                        $tipo_mensaje = "danger";
                        error_log("Error al crear área: " . $e->getMessage());
                    }
                }
                break;
                
            case 'editar':
                $id = $_POST['id'] ?? 0;
                $nombre_area = trim($_POST['nombre_area'] ?? '');
                $descripcion = trim($_POST['descripcion'] ?? '');
                $responsable = trim($_POST['responsable'] ?? '');
                $telefono_contacto = trim($_POST['telefono_contacto'] ?? '');
                
                if (empty($nombre_area)) {
                    $mensaje = "El nombre del área es obligatorio";
                    $tipo_mensaje = "danger";
                } else {
                    try {
                        // Verificar si el área ya existe (excluyendo el área actual)
                        $query = "SELECT id FROM areas WHERE nombre_area = :nombre AND id != :id";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':nombre', $nombre_area);
                        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                        $stmt->execute();
                        
                        if ($stmt->rowCount() > 0) {
                            $mensaje = "Ya existe otro área con ese nombre";
                            $tipo_mensaje = "danger";
                        } else {
                            $query = "UPDATE areas SET 
                                      nombre_area = :nombre,
                                      descripcion = :descripcion,
                                      responsable = :responsable,
                                      telefono_contacto = :telefono
                                      WHERE id = :id";
                            
                            $stmt = $db->prepare($query);
                            $stmt->bindParam(':nombre', $nombre_area);
                            $stmt->bindParam(':descripcion', $descripcion);
                            $stmt->bindParam(':responsable', $responsable);
                            $stmt->bindParam(':telefono', $telefono_contacto);
                            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                            
                            if ($stmt->execute()) {
                                $mensaje = "Área actualizada exitosamente";
                                $tipo_mensaje = "success";
                                $area_editar = null; // Limpiar edición
                            } else {
                                $mensaje = "Error al actualizar el área";
                                $tipo_mensaje = "danger";
                            }
                        }
                    } catch (PDOException $e) {
                        $mensaje = "Error: " . $e->getMessage();
                        $tipo_mensaje = "danger";
                        error_log("Error al editar área: " . $e->getMessage());
                    }
                }
                break;
                
            case 'eliminar':
                $id = $_POST['id'] ?? 0;
                
                try {
                    // Verificar si hay visitas asociadas
                    $query = "SELECT COUNT(*) as total FROM visitas WHERE area_id = :id";
                    $stmt = $db->prepare($query);
                    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                    $stmt->execute();
                    $result = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($result['total'] > 0) {
                        $mensaje = "No se puede eliminar el área porque tiene visitas asociadas";
                        $tipo_mensaje = "warning";
                    } else {
                        $query = "DELETE FROM areas WHERE id = :id";
                        $stmt = $db->prepare($query);
                        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                        
                        if ($stmt->execute()) {
                            $mensaje = "Área eliminada exitosamente";
                            $tipo_mensaje = "success";
                        } else {
                            $mensaje = "Error al eliminar el área";
                            $tipo_mensaje = "danger";
                        }
                    }
                } catch (PDOException $e) {
                    $mensaje = "Error: " . $e->getMessage();
                    $tipo_mensaje = "danger";
                }
                break;
                
            case 'toggle_estado':
                $id = $_POST['id'] ?? 0;
                
                try {
                    $query = "UPDATE areas SET activo = NOT activo WHERE id = :id";
                    $stmt = $db->prepare($query);
                    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                    
                    if ($stmt->execute()) {
                        $mensaje = "Estado del área actualizado";
                        $tipo_mensaje = "success";
                    }
                } catch (PDOException $e) {
                    $mensaje = "Error: " . $e->getMessage();
                    $tipo_mensaje = "danger";
                }
                break;
                
            case 'cargar_editar':
                $id = $_POST['id'] ?? 0;
                
                try {
                    $query = "SELECT * FROM areas WHERE id = :id";
                    $stmt = $db->prepare($query);
                    $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                    $stmt->execute();
                    
                    if ($stmt->rowCount() > 0) {
                        $area_editar = $stmt->fetch(PDO::FETCH_ASSOC);
                    }
                } catch (PDOException $e) {
                    error_log("Error al cargar área: " . $e->getMessage());
                }
                break;
        }
    }
}

// Obtener áreas
$areas = [];
$total_areas = 0;
$areas_activas = 0;

try {
    $query = "SELECT a.*, 
              (SELECT COUNT(*) FROM visitas v WHERE v.area_id = a.id) as total_visitas,
              (SELECT COUNT(*) FROM usuarios u WHERE u.area_id = a.id AND u.activo = 1) as total_usuarios
              FROM areas a 
              ORDER BY a.nombre_area";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    $areas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calcular estadísticas
    $total_areas = count($areas);
    foreach ($areas as $area) {
        if ($area['activo']) {
            $areas_activas++;
        }
    }
    
} catch (PDOException $e) {
    error_log("Error al cargar áreas: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Áreas - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .main-container {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 20px;
        }
        .card-header {
            background: linear-gradient(to right, #28a745, #20c997);
            color: white;
            border-radius: 15px 15px 0 0 !important;
            padding: 15px 20px;
            font-weight: 600;
        }
        .btn-primary {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            border: none;
            border-radius: 8px;
            padding: 10px 20px;
            transition: all 0.3s;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(33, 150, 243, 0.4);
        }
        .badge-estado {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.85em;
        }
        .badge-activo {
            background: #d4edda;
            color: #155724;
        }
        .badge-inactivo {
            background: #f8d7da;
            color: #721c24;
        }
        .tab-content {
            border: 1px solid #dee2e6;
            border-top: none;
            border-radius: 0 0 15px 15px;
            padding: 20px;
            background: white;
        }
        .nav-tabs .nav-link {
            border-radius: 15px 15px 0 0;
            margin-right: 5px;
        }
        .nav-tabs .nav-link.active {
            background: #28a745;
            color: white;
            border-color: #28a745;
        }
        .stat-card {
            background: white;
            border-radius: 10px;
            padding: 15px;
            text-align: center;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
            margin-bottom: 15px;
            transition: transform 0.3s;
        }
        .stat-card:hover {
            transform: translateY(-5px);
        }
        .stat-card .stat-number {
            font-size: 2rem;
            font-weight: bold;
            margin: 10px 0;
        }
        .stat-card.total {
            border-top: 5px solid var(--celeste-oscuro);
        }
        .stat-card.activas {
            border-top: 5px solid #28a745;
        }
        .area-icon {
            width: 50px;
            height: 50px;
            border-radius: 10px;
            background: linear-gradient(135deg, var(--celeste-primario), var(--celeste-secundario));
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 15px;
            color: var(--celeste-oscuro);
            font-size: 1.5rem;
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-container {
                margin-left: 0;
            }
        }
        #tablaAreas {
            border-radius: 10px;
            overflow: hidden;
        }
        #tablaAreas thead {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            color: white;
        }
        .table-hover tbody tr:hover {
            background-color: rgba(25, 118, 210, 0.05);
        }
        .badge-visitas {
            background: #e3f2fd;
            color: var(--celeste-oscuro);
            font-size: 0.8em;
            padding: 3px 8px;
            border-radius: 10px;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link" href="../../dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="../visitas/registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link" href="../visitas/salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link" href="../visitas/listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link" href="../reportes/generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link" href="../usuarios/gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link active" href="gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link" href="../entidad/configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="../../logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-container">
                <!-- Header -->
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Gestión de Áreas</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <span class="badge bg-primary">
                            <i class="fas fa-user-shield me-1"></i>Administrador
                        </span>
                    </div>
                </div>
                
                <?php if ($mensaje): ?>
                <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show" role="alert">
                    <i class="fas fa-<?php echo $tipo_mensaje == 'success' ? 'check-circle' : 'exclamation-circle'; ?> me-2"></i>
                    <?php echo $mensaje; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <!-- Estadísticas -->
                <div class="row mb-4">
                    <div class="col-md-4">
                        <div class="stat-card total">
                            <i class="fas fa-sitemap fa-2x text-primary mb-2"></i>
                            <h5>Total Áreas</h5>
                            <div class="stat-number text-primary"><?php echo $total_areas; ?></div>
                            <small>Registradas en el sistema</small>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="stat-card activas">
                            <i class="fas fa-check-circle fa-2x text-success mb-2"></i>
                            <h5>Áreas Activas</h5>
                            <div class="stat-number text-success"><?php echo $areas_activas; ?></div>
                            <small>Disponibles para visitas</small>
                        </div>
                    </div>
                    
                    <div class="col-md-4">
                        <div class="stat-card">
                            <i class="fas fa-users fa-2x text-warning mb-2"></i>
                            <h5>Porcentaje</h5>
                            <div class="stat-number text-warning">
                                <?php echo $total_areas > 0 ? round(($areas_activas / $total_areas) * 100, 1) : 0; ?>%
                            </div>
                            <small>Áreas activas</small>
                        </div>
                    </div>
                </div>
                
                <!-- Pestañas -->
                <ul class="nav nav-tabs" id="myTab" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="lista-tab" data-bs-toggle="tab" 
                                data-bs-target="#lista" type="button" role="tab">
                            <i class="fas fa-list me-2"></i>Lista de Áreas
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="nuevo-tab" data-bs-toggle="tab" 
                                data-bs-target="#nuevo" type="button" role="tab">
                            <i class="fas fa-plus-circle me-2"></i>
                            <?php echo $area_editar ? 'Editar Área' : 'Nueva Área'; ?>
                        </button>
                    </li>
                </ul>
                
                <div class="tab-content" id="myTabContent">
                    <!-- Tab Lista de Áreas -->
                    <div class="tab-pane fade show active" id="lista" role="tabpanel">
                        <?php if ($total_areas > 0): ?>
                        <div class="table-responsive">
                            <table id="tablaAreas" class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Área</th>
                                        <th>Descripción</th>
                                        <th>Responsable</th>
                                        <th>Contacto</th>
                                        <th>Estadísticas</th>
                                        <th>Estado</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($areas as $area): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="area-icon">
                                                    <i class="fas fa-building"></i>
                                                </div>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($area['nombre_area']); ?></strong>
                                                    <br>
                                                    <small class="text-muted">
                                                        ID: <?php echo $area['id']; ?>
                                                    </small>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <small>
                                                <?php 
                                                if (!empty($area['descripcion'])) {
                                                    echo htmlspecialchars(substr($area['descripcion'], 0, 50));
                                                    if (strlen($area['descripcion']) > 50) echo '...';
                                                } else {
                                                    echo '<span class="text-muted">Sin descripción</span>';
                                                }
                                                ?>
                                            </small>
                                        </td>
                                        <td>
                                            <?php 
                                            if (!empty($area['responsable'])) {
                                                echo htmlspecialchars($area['responsable']);
                                            } else {
                                                echo '<span class="text-muted">No asignado</span>';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <?php 
                                            if (!empty($area['telefono_contacto'])) {
                                                echo '<i class="fas fa-phone me-1"></i>' . htmlspecialchars($area['telefono_contacto']);
                                            } else {
                                                echo '<span class="text-muted">No disponible</span>';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <div class="d-flex flex-wrap gap-2">
                                                <span class="badge-visitas" data-bs-toggle="tooltip" 
                                                      title="Total de visitas">
                                                    <i class="fas fa-users me-1"></i>
                                                    <?php echo $area['total_visitas'] ?? 0; ?>
                                                </span>
                                                <span class="badge-visitas" data-bs-toggle="tooltip" 
                                                      title="Usuarios asignados">
                                                    <i class="fas fa-user me-1"></i>
                                                    <?php echo $area['total_usuarios'] ?? 0; ?>
                                                </span>
                                            </div>
                                        </td>
                                        <td>
                                            <span class="badge-estado <?php echo $area['activo'] ? 'badge-activo' : 'badge-inactivo'; ?>">
                                                <i class="fas fa-<?php echo $area['activo'] ? 'check-circle' : 'times-circle'; ?> me-1"></i>
                                                <?php echo $area['activo'] ? 'Activa' : 'Inactiva'; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <!-- Editar -->
                                                <form method="POST" action="" style="display: inline;">
                                                    <input type="hidden" name="accion" value="cargar_editar">
                                                    <input type="hidden" name="id" value="<?php echo $area['id']; ?>">
                                                    <button type="submit" class="btn btn-outline-primary btn-sm" 
                                                            data-bs-toggle="tooltip" title="Editar área">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                </form>
                                                
                                                <!-- Activar/Desactivar -->
                                                <form method="POST" action="" style="display: inline;">
                                                    <input type="hidden" name="accion" value="toggle_estado">
                                                    <input type="hidden" name="id" value="<?php echo $area['id']; ?>">
                                                    <button type="submit" class="btn btn-outline-<?php echo $area['activo'] ? 'warning' : 'success'; ?> btn-sm"
                                                            data-bs-toggle="tooltip" 
                                                            title="<?php echo $area['activo'] ? 'Desactivar área' : 'Activar área'; ?>">
                                                        <i class="fas fa-<?php echo $area['activo'] ? 'ban' : 'check'; ?>"></i>
                                                    </button>
                                                </form>
                                                
                                                <!-- Eliminar -->
                                                <form method="POST" action="" style="display: inline;" 
                                                      onsubmit="return confirm('¿Está seguro de eliminar esta área?');">
                                                    <input type="hidden" name="accion" value="eliminar">
                                                    <input type="hidden" name="id" value="<?php echo $area['id']; ?>">
                                                    <button type="submit" class="btn btn-outline-danger btn-sm"
                                                            data-bs-toggle="tooltip" title="Eliminar área"
                                                            <?php echo (($area['total_visitas'] ?? 0) > 0) ? 'disabled' : ''; ?>>
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                <strong><?php echo $total_areas; ?> áreas registradas.</strong> 
                                Las áreas con visitas asociadas no pueden ser eliminadas.
                            </div>
                        </div>
                        <?php else: ?>
                        <div class="text-center py-5">
                            <i class="fas fa-sitemap fa-4x text-muted mb-3"></i>
                            <h4 class="mb-3">No hay áreas registradas</h4>
                            <p class="text-muted mb-4">Debe crear al menos una área para poder registrar visitas.</p>
                            <button class="btn btn-primary" onclick="cambiarTab('nuevo')">
                                <i class="fas fa-plus-circle me-2"></i>Crear Primera Área
                            </button>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Tab Nueva/Editar Área -->
                    <div class="tab-pane fade" id="nuevo" role="tabpanel">
                        <form method="POST" action="" id="formArea">
                            <?php if ($area_editar): ?>
                                <input type="hidden" name="accion" value="editar">
                                <input type="hidden" name="id" value="<?php echo $area_editar['id']; ?>">
                            <?php else: ?>
                                <input type="hidden" name="accion" value="crear">
                            <?php endif; ?>
                            
                            <div class="row">
                                <div class="col-md-12 mb-3">
                                    <label for="nombre_area" class="form-label">Nombre del Área *</label>
                                    <input type="text" class="form-control" id="nombre_area" name="nombre_area" 
                                           value="<?php echo htmlspecialchars($area_editar['nombre_area'] ?? ($_POST['nombre_area'] ?? '')); ?>" 
                                           required placeholder="Ej: Administración, Recursos Humanos, etc.">
                                    <small class="text-muted">Este nombre aparecerá en los registros de visitas</small>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="responsable" class="form-label">Responsable</label>
                                    <input type="text" class="form-control" id="responsable" name="responsable" 
                                           value="<?php echo htmlspecialchars($area_editar['responsable'] ?? ($_POST['responsable'] ?? '')); ?>" 
                                           placeholder="Nombre del responsable del área">
                                </div>
                                
                                <div class="col-md-6 mb-3">
                                    <label for="telefono_contacto" class="form-label">Teléfono de Contacto</label>
                                    <input type="tel" class="form-control" id="telefono_contacto" name="telefono_contacto" 
                                           value="<?php echo htmlspecialchars($area_editar['telefono_contacto'] ?? ($_POST['telefono_contacto'] ?? '')); ?>" 
                                           placeholder="Ej: +51 123456789">
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="descripcion" class="form-label">Descripción</label>
                                <textarea class="form-control" id="descripcion" name="descripcion" 
                                          rows="3" placeholder="Breve descripción del área y sus funciones"><?php echo htmlspecialchars($area_editar['descripcion'] ?? ($_POST['descripcion'] ?? '')); ?></textarea>
                            </div>
                            
                            <div class="d-flex justify-content-between mt-4">
                                <?php if ($area_editar): ?>
                                    <a href="gestion.php" class="btn btn-secondary">
                                        <i class="fas fa-times me-1"></i> Cancelar Edición
                                    </a>
                                <?php else: ?>
                                    <button type="reset" class="btn btn-secondary">
                                        <i class="fas fa-redo me-1"></i> Limpiar Formulario
                                    </button>
                                <?php endif; ?>
                                
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-1"></i>
                                    <?php echo $area_editar ? 'Actualizar Área' : 'Crear Área'; ?>
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    
    <script>
        // Inicializar DataTable
        $(document).ready(function() {
            $('#tablaAreas').DataTable({
                language: {
                    url: '//cdn.datatables.net/plug-ins/1.11.5/i18n/es-ES.json'
                },
                pageLength: 25,
                order: [[0, 'asc']],
                dom: '<"row"<"col-md-6"l><"col-md-6"f>>rt<"row"<"col-md-6"i><"col-md-6"p>>',
                columnDefs: [
                    { orderable: false, targets: [6] } // Columna de acciones no ordenable
                ]
            });
            
            // Inicializar tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
            
            // Cambiar a pestaña de edición si hay área para editar
            <?php if ($area_editar): ?>
            cambiarTab('nuevo');
            <?php endif; ?>
        });
        
        // Función para cambiar de pestaña
        function cambiarTab(tabId) {
            var tab = new bootstrap.Tab(document.getElementById(tabId + '-tab'));
            tab.show();
        }
        
        // Validar formulario de área
        document.getElementById('formArea').addEventListener('submit', function(e) {
            var nombre = document.getElementById('nombre_area').value.trim();
            
            if (nombre === '') {
                e.preventDefault();
                alert('El nombre del área es obligatorio');
                return false;
            }
            
            if (nombre.length < 3) {
                e.preventDefault();
                alert('El nombre del área debe tener al menos 3 caracteres');
                return false;
            }
            
            return true;
        });
        
        // Mostrar advertencia si se intenta eliminar área con visitas
        document.addEventListener('submit', function(e) {
            if (e.target && e.target.querySelector('input[name="accion"][value="eliminar"]')) {
                var button = e.target.querySelector('button[type="submit"]');
                if (button && button.disabled) {
                    e.preventDefault();
                    alert('Esta área tiene visitas asociadas y no puede ser eliminada. Puede desactivarla en su lugar.');
                    return false;
                }
            }
        });
    </script>
</body>
</html>