<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Verificar sesión
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

require_once '../../config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

// Variables para filtros
$fecha_inicio = $_GET['fecha_inicio'] ?? date('Y-m-d', strtotime('-30 days'));
$fecha_fin = $_GET['fecha_fin'] ?? date('Y-m-d');
$dni = $_GET['dni'] ?? '';
$estado = $_GET['estado'] ?? 'todos';

// Obtener áreas para filtro
$areas = [];
try {
    $query = "SELECT id, nombre_area FROM areas WHERE activo = 1 ORDER BY nombre_area";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $areas = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar áreas: " . $e->getMessage());
}

$area_id = $_GET['area_id'] ?? 'todas';

// Obtener visitas con filtros
$visitas = [];
$total_visitas = 0;
$total_activas = 0;

try {
    $query = "SELECT v.*, a.nombre_area as area_nombre, 
              CONCAT(u.nombres, ' ', u.apellidos) as usuario_registra
              FROM visitas v 
              LEFT JOIN areas a ON v.area_id = a.id 
              LEFT JOIN usuarios u ON v.usuario_registra_id = u.id 
              WHERE DATE(v.fecha_visita) BETWEEN :fecha_inicio AND :fecha_fin";
    
    $params = [
        ':fecha_inicio' => $fecha_inicio,
        ':fecha_fin' => $fecha_fin
    ];
    
    // Aplicar filtros adicionales
    if (!empty($dni)) {
        $query .= " AND v.documento_visitante LIKE :dni";
        $params[':dni'] = "%$dni%";
    }
    
    if (!empty($area_id) && $area_id != 'todas') {
        $query .= " AND v.area_id = :area_id";
        $params[':area_id'] = $area_id;
    }
    
    // Filtro por estado
    if ($estado == 'activas') {
        $query .= " AND v.hora_salida IS NULL";
    } elseif ($estado == 'completadas') {
        $query .= " AND v.hora_salida IS NOT NULL";
    }
    
    $query .= " ORDER BY v.fecha_visita DESC, v.hora_ingreso DESC";
    
    $stmt = $db->prepare($query);
    
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    
    $stmt->execute();
    $visitas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Calcular estadísticas
    $total_visitas = count($visitas);
    foreach ($visitas as $visita) {
        if ($visita['hora_salida'] === null) {
            $total_activas++;
        }
    }
    
} catch (PDOException $e) {
    error_log("Error al cargar visitas: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Listar Visitas - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .main-container {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 20px;
        }
        .card-header {
            background: linear-gradient(to right, #28a745, #20c997);
            color: white;
            border-radius: 15px 15px 0 0 !important;
            padding: 15px 20px;
            font-weight: 600;
        }
        .btn-primary {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            border: none;
            border-radius: 8px;
            padding: 10px 20px;
            transition: all 0.3s;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(33, 150, 243, 0.4);
        }
        .badge-estado {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.85em;
        }
        .badge-activa {
            background: #fff3cd;
            color: #856404;
        }
        .badge-completada {
            background: #d4edda;
            color: #155724;
        }
        .filter-card {
            background: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
        }
        .stat-badge {
            font-size: 0.9em;
            padding: 8px 15px;
            border-radius: 20px;
            margin-right: 10px;
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-container {
                margin-left: 0;
            }
        }
        #tablaVisitas {
            border-radius: 10px;
            overflow: hidden;
        }
        #tablaVisitas thead {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            color: white;
        }
        .table-hover tbody tr:hover {
            background-color: rgba(25, 118, 210, 0.05);
        }
        .text-muted-small {
            font-size: 0.85em;
            color: #6c757d;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link" href="../../dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link" href="salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link active" href="listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link" href="../reportes/generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <?php if (isset($_SESSION['user_tipo']) && $_SESSION['user_tipo'] == 'admin'): ?>
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link" href="../usuarios/gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link" href="../areas/gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link" href="../entidad/configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        <?php endif; ?>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="../../logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-container">
                <!-- Header -->
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Listado de Visitas</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <span class="badge bg-primary me-2">
                            <i class="fas fa-user me-1"></i>
                            <?php echo isset($_SESSION['user_nombre']) ? htmlspecialchars($_SESSION['user_nombre']) : 'Usuario'; ?>
                        </span>
                        <div class="btn-group">
                            <a href="registrar.php" class="btn btn-sm btn-success">
                                <i class="fas fa-plus-circle me-1"></i>Nueva Visita
                            </a>
                            <a href="salida.php" class="btn btn-sm btn-warning">
                                <i class="fas fa-sign-out-alt me-1"></i>Registrar Salida
                            </a>
                        </div>
                    </div>
                </div>
                
                <!-- Estadísticas -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <span class="badge bg-primary stat-badge">
                            <i class="fas fa-users me-1"></i>Total: <?php echo $total_visitas; ?>
                        </span>
                    </div>
                    <div class="col-md-3">
                        <span class="badge bg-warning stat-badge">
                            <i class="fas fa-user-clock me-1"></i>Activas: <?php echo $total_activas; ?>
                        </span>
                    </div>
                    <div class="col-md-3">
                        <span class="badge bg-success stat-badge">
                            <i class="fas fa-check-circle me-1"></i>Completadas: <?php echo $total_visitas - $total_activas; ?>
                        </span>
                    </div>
                    <div class="col-md-3 text-end">
                        <span class="text-muted-small">
                            <i class="fas fa-calendar me-1"></i>
                            <?php echo date('d/m/Y', strtotime($fecha_inicio)); ?> - <?php echo date('d/m/Y', strtotime($fecha_fin)); ?>
                        </span>
                    </div>
                </div>
                
                <!-- Filtros -->
                <div class="filter-card">
                    <h5 class="mb-3"><i class="fas fa-filter me-2"></i>Filtros de Búsqueda</h5>
                    <form method="GET" action="" id="formFiltros">
                        <div class="row">
                            <div class="col-md-3 mb-3">
                                <label for="fecha_inicio" class="form-label">Fecha Inicio</label>
                                <input type="date" class="form-control" id="fecha_inicio" name="fecha_inicio" 
                                       value="<?php echo htmlspecialchars($fecha_inicio); ?>" required>
                            </div>
                            
                            <div class="col-md-3 mb-3">
                                <label for="fecha_fin" class="form-label">Fecha Fin</label>
                                <input type="date" class="form-control" id="fecha_fin" name="fecha_fin" 
                                       value="<?php echo htmlspecialchars($fecha_fin); ?>" required>
                            </div>
                            
                            <div class="col-md-2 mb-3">
                                <label for="dni" class="form-label">DNI Visitante</label>
                                <input type="text" class="form-control" id="dni" name="dni" 
                                       value="<?php echo htmlspecialchars($dni); ?>" 
                                       placeholder="Buscar por DNI">
                            </div>
                            
                            <div class="col-md-2 mb-3">
                                <label for="area_id" class="form-label">Área</label>
                                <select class="form-select" id="area_id" name="area_id">
                                    <option value="todas">Todas las áreas</option>
                                    <?php foreach ($areas as $area): ?>
                                        <option value="<?php echo $area['id']; ?>" 
                                            <?php echo ($area_id == $area['id']) ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($area['nombre_area']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="col-md-2 mb-3">
                                <label for="estado" class="form-label">Estado</label>
                                <select class="form-select" id="estado" name="estado">
                                    <option value="todos" <?php echo ($estado == 'todos') ? 'selected' : ''; ?>>Todos</option>
                                    <option value="activas" <?php echo ($estado == 'activas') ? 'selected' : ''; ?>>Solo activas</option>
                                    <option value="completadas" <?php echo ($estado == 'completadas') ? 'selected' : ''; ?>>Solo completadas</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-12">
                                <div class="btn-group">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search me-2"></i>Buscar
                                    </button>
                                    <a href="listar.php" class="btn btn-secondary">
                                        <i class="fas fa-redo me-2"></i>Limpiar
                                    </a>
                                    <!-- 
                                    <button type="button" class="btn btn-success" onclick="exportarExcel()">
                                        <i class="fas fa-file-excel me-2"></i>Exportar
                                    </button> 
                                    -->
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                
                <!-- Tabla de visitas -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-table me-2"></i>Visitas Registradas
                            <small class="text-white-50 ms-2">(<?php echo $total_visitas; ?> registros)</small>
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if ($total_visitas > 0): ?>
                        <div class="table-responsive">
                            <table id="tablaVisitas" class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Fecha</th>
                                        <th>Visitante</th>
                                        <th>DNI</th>
                                        <th>Área</th>
                                        <th>Hora Ingreso</th>
                                        <th>Hora Salida</th>
                                        <th>Estado</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($visitas as $visita): 
                                        // Calcular tiempo transcurrido si está activa
                                        $tiempo_transcurrido = '';
                                        $badge_class = 'badge-completada';
                                        $icono = 'check-circle';
                                        $texto = 'Completada';
                                        
                                        if ($visita['hora_salida'] === null) {
                                            $badge_class = 'badge-activa';
                                            $icono = 'clock';
                                            $texto = 'Activa';
                                            
                                            // Calcular tiempo transcurrido
                                            $hora_ingreso = strtotime($visita['hora_ingreso']);
                                            $tiempo_transcurrido = time() - $hora_ingreso;
                                            $horas = floor($tiempo_transcurrido / 3600);
                                            $minutos = floor(($tiempo_transcurrido % 3600) / 60);
                                            $texto .= ' (' . sprintf("%02d:%02d", $horas, $minutos) . ')';
                                        }
                                    ?>
                                    <tr>
                                        <td>
                                            <?php echo date('d/m/Y', strtotime($visita['fecha_visita'])); ?>
                                            <br>
                                            <small class="text-muted-small">
                                                <?php echo date('D', strtotime($visita['fecha_visita'])); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <strong><?php echo htmlspecialchars($visita['apellidos_visitante']); ?></strong>
                                            <br>
                                            <small><?php echo htmlspecialchars($visita['nombres_visitante']); ?></small>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <?php echo htmlspecialchars($visita['documento_visitante']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo htmlspecialchars($visita['area_nombre'] ?? 'No especificada'); ?>
                                            <?php if (!empty($visita['usuario_registra'])): ?>
                                            <br>
                                            <small class="text-muted-small">
                                                <i class="fas fa-user-edit me-1"></i>
                                                <?php echo htmlspecialchars($visita['usuario_registra']); ?>
                                            </small>
                                            <?php endif; ?>
                                        </td>
                                        <td class="fw-bold">
                                            <?php echo date('H:i', strtotime($visita['hora_ingreso'])); ?>
                                            <br>
                                            <small class="text-muted-small">
                                                <?php echo $visita['tipo_ingreso'] == 'manual' ? '(M)' : '(A)'; ?>
                                            </small>
                                        </td>
                                        <td>
                                            <?php if ($visita['hora_salida']): ?>
                                                <span class="text-success fw-bold">
                                                    <?php echo date('H:i', strtotime($visita['hora_salida'])); ?>
                                                </span>
                                                <br>
                                                <small class="text-muted-small">
                                                    <?php echo $visita['tipo_salida'] == 'manual' ? '(M)' : '(A)'; ?>
                                                </small>
                                            <?php else: ?>
                                                <span class="text-warning">Pendiente</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <span class="badge badge-estado <?php echo $badge_class; ?>">
                                                <i class="fas fa-<?php echo $icono; ?> me-1"></i>
                                                <?php echo $texto; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <button type="button" class="btn btn-outline-info" 
                                                        data-bs-toggle="tooltip" title="Ver detalles"
                                                        onclick="verDetalle(<?php echo $visita['id']; ?>)">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                
                                                <?php if ($visita['hora_salida'] === null): ?>
                                                <button type="button" class="btn btn-outline-warning" 
                                                        data-bs-toggle="tooltip" title="Registrar salida"
                                                        onclick="registrarSalida(<?php echo $visita['id']; ?>)">
                                                    <i class="fas fa-sign-out-alt"></i>
                                                </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3 text-muted">
                            <small>
                                <i class="fas fa-info-circle me-1"></i>
                                Mostrando <?php echo $total_visitas; ?> registros.
                                <span class="ms-2">
                                    <i class="fas fa-clock me-1"></i>
                                    (A) = Automático, (M) = Manual
                                </span>
                            </small>
                        </div>
                        <?php else: ?>
                        <div class="text-center py-5">
                            <i class="fas fa-search fa-4x text-muted mb-3"></i>
                            <h4 class="mb-3">No se encontraron visitas</h4>
                            <p class="text-muted mb-4">No hay visitas que coincidan con los criterios de búsqueda.</p>
                            <a href="listar.php" class="btn btn-primary me-2">
                                <i class="fas fa-redo me-2"></i>Ver todas las visitas
                            </a>
                            <a href="registrar.php" class="btn btn-success">
                                <i class="fas fa-plus-circle me-2"></i>Registrar Nueva Visita
                            </a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal para ver detalles -->
    <div class="modal fade" id="modalDetalle" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="fas fa-info-circle me-2"></i>Detalles de la Visita</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="detalleContenido">
                    <!-- Los detalles se cargarán aquí por AJAX -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
    
    <script>
        // Inicializar DataTable
        $(document).ready(function() {
            $('#tablaVisitas').DataTable({
                language: {
                    url: '//cdn.datatables.net/plug-ins/1.11.5/i18n/es-ES.json'
                },
                pageLength: 25,
                order: [[0, 'desc']],
                dom: '<"row"<"col-md-6"l><"col-md-6"f>>rt<"row"<"col-md-6"i><"col-md-6"p>>',
                columnDefs: [
                    { orderable: false, targets: [7] } // Columna de acciones no ordenable
                ]
            });
            
            // Inicializar tooltips
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        });
        
        // Función para ver detalles de la visita
        function verDetalle(id) {
            // Mostrar loading
            $('#detalleContenido').html(`
                <div class="text-center py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Cargando...</span>
                    </div>
                    <p class="mt-2">Cargando detalles...</p>
                </div>
            `);
            
            // Cargar detalles por AJAX
            $.ajax({
                url: 'detalle_visita.php',
                method: 'GET',
                data: { id: id },
                success: function(response) {
                    $('#detalleContenido').html(response);
                },
                error: function() {
                    $('#detalleContenido').html(`
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            Error al cargar los detalles de la visita.
                        </div>
                    `);
                }
            });
            
            // Mostrar modal
            var modal = new bootstrap.Modal(document.getElementById('modalDetalle'));
            modal.show();
        }
        
        // Función para registrar salida
        function registrarSalida(id) {
            if (confirm('¿Desea registrar la salida de esta visita?')) {
                // Redirigir a la página de salida
                window.location.href = 'salida.php?visita_id=' + id;
            }
        }
        
        // Función para exportar a Excel
        function exportarExcel() {
            // Crear una nueva hoja de trabajo
            var wb = XLSX.utils.book_new();
            
            // Preparar datos
            var data = [
                ['Fecha', 'Visitante', 'DNI', 'Área', 'Hora Ingreso', 'Hora Salida', 'Motivo', 'Estado', 'Registrado por']
            ];
            
            // Obtener datos de la tabla
            var table = document.getElementById('tablaVisitas');
            var rows = table.querySelectorAll('tbody tr');
            
            rows.forEach(function(row) {
                var cells = row.querySelectorAll('td');
                if (cells.length >= 7) {
                    var fecha = cells[0].textContent.split('\n')[0].trim();
                    var visitante = cells[1].textContent.replace(/\n/g, ' ').trim();
                    var dni = cells[2].textContent.trim();
                    var area = cells[3].textContent.split('\n')[0].trim();
                    var ingreso = cells[4].textContent.split('\n')[0].trim();
                    var salida = cells[5].textContent.split('\n')[0].trim();
                    var estado = cells[6].textContent.trim();
                    
                    data.push([
                        fecha, visitante, dni, area, ingreso, salida, '', estado, ''
                    ]);
                }
            });
            
            // Crear hoja de cálculo
            var ws = XLSX.utils.aoa_to_sheet(data);
            
            // Añadir hoja al libro
            XLSX.utils.book_append_sheet(wb, ws, "Visitas");
            
            // Generar nombre de archivo
            var fecha = new Date().toISOString().slice(0,10);
            var filename = "visitas_" + fecha + ".xlsx";
            
            // Descargar archivo
            XLSX.writeFile(wb, filename);
        }
        
        // Validar fechas en el formulario de filtros
        document.getElementById('formFiltros').addEventListener('submit', function(e) {
            var fechaInicio = new Date(document.getElementById('fecha_inicio').value);
            var fechaFin = new Date(document.getElementById('fecha_fin').value);
            var hoy = new Date();
            hoy.setHours(0,0,0,0);
            
            if (fechaFin < fechaInicio) {
                e.preventDefault();
                alert('La fecha fin no puede ser menor a la fecha inicio');
                return false;
            }
            
            if (fechaInicio > hoy) {
                e.preventDefault();
                alert('La fecha inicio no puede ser futura');
                return false;
            }
            
            if (fechaFin > hoy) {
                e.preventDefault();
                alert('La fecha fin no puede ser futura');
                return false;
            }
            
            // Limitar rango a máximo 90 días
            var diffTime = Math.abs(fechaFin - fechaInicio);
            var diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
            
            if (diffDays > 90) {
                e.preventDefault();
                alert('El rango de fechas no puede superar los 90 días');
                return false;
            }
            
            return true;
        });
        
        // Setear fecha máxima como hoy
        document.addEventListener('DOMContentLoaded', function() {
            var hoy = new Date().toISOString().split('T')[0];
            document.getElementById('fecha_inicio').max = hoy;
            document.getElementById('fecha_fin').max = hoy;
        });
        
        // Actualizar tiempos de visitas activas cada minuto
        function actualizarTiemposActivas() {
            var badgesActivas = document.querySelectorAll('.badge-activa');
            
            badgesActivas.forEach(function(badge) {
                var texto = badge.textContent;
                var match = texto.match(/Activa \((\d{2}):(\d{2})\)/);
                
                if (match) {
                    var horas = parseInt(match[1]);
                    var minutos = parseInt(match[2]);
                    
                    // Incrementar minutos
                    minutos++;
                    if (minutos >= 60) {
                        minutos = 0;
                        horas++;
                    }
                    
                    // Actualizar texto
                    badge.textContent = 'Activa (' + 
                        horas.toString().padStart(2, '0') + ':' + 
                        minutos.toString().padStart(2, '0') + ')';
                }
            });
        }
        
        // Actualizar cada minuto si hay visitas activas
        <?php if ($total_activas > 0): ?>
        setInterval(actualizarTiemposActivas, 60000);
        <?php endif; ?>
    </script>
</body>
</html>