<?php
// Habilitar errores temporalmente
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();

// Verificar sesión
if (!isset($_SESSION['user_id'])) {
    header("Location: ../../login.php");
    exit();
}

require_once '../../config/database.php';

$database = new Database();
$db = $database->getConnection();

if (!$db) {
    die("Error de conexión a la base de datos");
}

$mensaje = '';
$tipo_mensaje = '';
$visitas_activas = [];

// Procesar registro de salida
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['visita_id'])) {
    $visita_id = $_POST['visita_id'];
    $tipo_salida = $_POST['tipo_salida'] ?? 'automatico';
    
    // Determinar hora de salida
    if ($tipo_salida == 'manual' && isset($_POST['hora_salida'])) {
        $hora_salida = $_POST['hora_salida'];
        // Asegurar formato HH:MM:SS
        if (strlen($hora_salida) == 5) {
            $hora_salida .= ':00';
        }
    } else {
        $hora_salida = date('H:i:s');
    }
    
    try {
        $query = "UPDATE visitas SET hora_salida = :hora_salida, tipo_salida = :tipo_salida 
                  WHERE id = :id AND hora_salida IS NULL";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':hora_salida', $hora_salida);
        $stmt->bindParam(':tipo_salida', $tipo_salida);
        $stmt->bindParam(':id', $visita_id, PDO::PARAM_INT);
        
        if ($stmt->execute()) {
            if ($stmt->rowCount() > 0) {
                $mensaje = "Salida registrada exitosamente!";
                $tipo_mensaje = "success";
                
                // Mostrar mensaje con JavaScript para recargar la página
                echo "<script>
                    setTimeout(function() {
                        window.location.reload();
                    }, 1500);
                </script>";
            } else {
                $mensaje = "La visita ya tiene salida registrada o no existe.";
                $tipo_mensaje = "warning";
            }
        } else {
            $mensaje = "Error al registrar la salida.";
            $tipo_mensaje = "danger";
        }
    } catch (PDOException $e) {
        $mensaje = "Error: " . $e->getMessage();
        $tipo_mensaje = "danger";
        error_log("Error en registro de salida: " . $e->getMessage());
    }
}

// Obtener visitas activas (sin salida registrada) del día de hoy
try {
    $query = "SELECT v.*, a.nombre_area as area, 
              CONCAT(v.apellidos_visitante, ', ', v.nombres_visitante) as visitante
              FROM visitas v 
              LEFT JOIN areas a ON v.area_id = a.id 
              WHERE v.hora_salida IS NULL 
              AND DATE(v.fecha_visita) = CURDATE() 
              ORDER BY v.hora_ingreso DESC";
    
    $stmt = $db->prepare($query);
    $stmt->execute();
    $visitas_activas = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Error al cargar visitas activas: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Registrar Salida - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        :root {
            --celeste-primario: #e3f2fd;
            --celeste-secundario: #bbdefb;
            --celeste-oscuro: #1976d2;
        }
        body {
            background-color: #f8f9fa;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .main-container {
            margin-left: 250px;
            padding: 20px;
            min-height: 100vh;
        }
        .sidebar {
            background: linear-gradient(180deg, var(--celeste-oscuro) 0%, #1565c0 100%);
            min-height: 100vh;
            color: white;
            padding: 0;
            position: fixed;
            width: 250px;
        }
        .sidebar .nav-link {
            color: rgba(255,255,255,0.8);
            padding: 12px 20px;
            margin: 2px 10px;
            border-radius: 8px;
            transition: all 0.3s;
            text-decoration: none;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            color: white;
            transform: translateX(5px);
        }
        .card {
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            border: none;
            margin-bottom: 20px;
        }
        .card-header {
            background: linear-gradient(to right, #ff9800, #ffb74d);
            color: white;
            border-radius: 15px 15px 0 0 !important;
            padding: 15px 20px;
            font-weight: 600;
        }
        .badge-tiempo {
            font-size: 0.9em;
            padding: 5px 10px;
            border-radius: 20px;
        }
        .table-custom {
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 3px 10px rgba(0,0,0,0.05);
        }
        .table-custom thead {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            color: white;
        }
        .table-custom tbody tr:hover {
            background-color: #f8f9fa;
        }
        .btn-warning {
            background: linear-gradient(to right, #ff9800, #ffb74d);
            border: none;
            border-radius: 8px;
            padding: 8px 15px;
            transition: all 0.3s;
        }
        .btn-warning:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(255, 152, 0, 0.4);
        }
        .btn-primary {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            border: none;
            border-radius: 8px;
        }
        @media (max-width: 768px) {
            .sidebar {
                width: 100%;
                position: relative;
                min-height: auto;
            }
            .main-container {
                margin-left: 0;
            }
        }
        .modal-content {
            border-radius: 15px;
            border: none;
        }
        .modal-header {
            background: linear-gradient(to right, var(--celeste-oscuro), #2196f3);
            color: white;
            border-radius: 15px 15px 0 0;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row">
            <!-- Sidebar -->
            <div class="col-md-3 col-lg-2 d-md-block sidebar">
                <div class="position-sticky pt-3">
                    <div class="text-center mb-4 p-3">
                        <div class="p-3">
                            <i class="fas fa-building fa-3x mb-3"></i>
                            <h5>Sistema de Visitas</h5>
                            <small>Entidad Pública</small>
                        </div>
                    </div>
                    
                    <div class="nav flex-column">
                        <a class="nav-link" href="../../dashboard.php">
                            <i class="fas fa-home me-2"></i>Dashboard
                        </a>
                        <a class="nav-link" href="registrar.php">
                            <i class="fas fa-sign-in-alt me-2"></i>Registrar Ingreso
                        </a>
                        <a class="nav-link active" href="salida.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                        </a>
                        <a class="nav-link" href="listar.php">
                            <i class="fas fa-list me-2"></i>Listar Visitas
                        </a>
                        <a class="nav-link" href="../reportes/generar.php">
                            <i class="fas fa-chart-bar me-2"></i>Reportes
                        </a>
                        
                        <?php if (isset($_SESSION['user_tipo']) && $_SESSION['user_tipo'] == 'admin'): ?>
                        <hr class="my-3 text-white-50">
                        <h6 class="sidebar-heading px-3 mt-4 mb-1 text-white-50">
                            <span>Administración</span>
                        </h6>
                        <a class="nav-link" href="../usuarios/gestion.php">
                            <i class="fas fa-users me-2"></i>Gestión de Usuarios
                        </a>
                        <a class="nav-link" href="../areas/gestion.php">
                            <i class="fas fa-sitemap me-2"></i>Áreas
                        </a>
                        <a class="nav-link" href="../entidad/configuracion.php">
                            <i class="fas fa-cog me-2"></i>Configuración
                        </a>
                        <?php endif; ?>
                        
                        <hr class="my-3 text-white-50">
                        <a class="nav-link" href="../../logout.php">
                            <i class="fas fa-sign-out-alt me-2"></i>Cerrar Sesión
                        </a>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="col-md-9 ms-sm-auto col-lg-10 px-md-4 main-container">
                <!-- Header -->
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">Registrar Salida de Visitantes</h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <span class="badge bg-warning">
                            <i class="fas fa-users me-1"></i>
                            <?php echo count($visitas_activas); ?> visitas activas
                        </span>
                        <span class="badge bg-primary ms-2">
                            <i class="fas fa-user me-1"></i>
                            <?php echo isset($_SESSION['user_nombre']) ? htmlspecialchars($_SESSION['user_nombre']) : 'Usuario'; ?>
                        </span>
                    </div>
                </div>
                
                <?php if ($mensaje): ?>
                <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show" role="alert">
                    <i class="fas fa-<?php echo $tipo_mensaje == 'success' ? 'check-circle' : 'exclamation-circle'; ?> me-2"></i>
                    <?php echo $mensaje; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <?php if (empty($visitas_activas)): ?>
                <div class="card">
                    <div class="card-body text-center py-5">
                        <i class="fas fa-check-circle fa-4x text-success mb-3"></i>
                        <h4 class="mb-3">No hay visitas activas</h4>
                        <p class="text-muted mb-4">No hay visitantes pendientes de salida para el día de hoy.</p>
                        <a href="registrar.php" class="btn btn-primary">
                            <i class="fas fa-plus-circle me-2"></i>Registrar Nueva Visita
                        </a>
                    </div>
                </div>
                <?php else: ?>
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-list me-2"></i>Visitas Activas - <?php echo date('d/m/Y'); ?>
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover table-custom">
                                <thead>
                                    <tr>
                                        <th>Hora Ingreso</th>
                                        <th>Visitante</th>
                                        <th>DNI</th>
                                        <th>Área</th>
                                        <th>Motivo</th>
                                        <th>Tiempo Transcurrido</th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($visitas_activas as $visita): 
                                        // Calcular tiempo transcurrido
                                        $hora_ingreso = strtotime($visita['hora_ingreso']);
                                        $tiempo_transcurrido = time() - $hora_ingreso;
                                        $horas = floor($tiempo_transcurrido / 3600);
                                        $minutos = floor(($tiempo_transcurrido % 3600) / 60);
                                        
                                        // Determinar color del badge según el tiempo
                                        $badge_class = 'bg-success';
                                        if ($horas >= 2) {
                                            $badge_class = 'bg-danger';
                                        } elseif ($horas >= 1) {
                                            $badge_class = 'bg-warning';
                                        }
                                    ?>
                                    <tr>
                                        <td class="fw-bold"><?php echo date('H:i', $hora_ingreso); ?></td>
                                        <td><?php echo htmlspecialchars($visita['visitante']); ?></td>
                                        <td><span class="badge bg-secondary"><?php echo $visita['documento_visitante']; ?></span></td>
                                        <td><?php echo htmlspecialchars($visita['area'] ?? 'No especificada'); ?></td>
                                        <td>
                                            <span data-bs-toggle="tooltip" title="<?php echo htmlspecialchars($visita['motivo_visita']); ?>">
                                                <?php echo htmlspecialchars(substr($visita['motivo_visita'], 0, 30)) . '...'; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge <?php echo $badge_class; ?> badge-tiempo">
                                                <i class="fas fa-clock me-1"></i>
                                                <?php echo sprintf("%02d:%02d", $horas, $minutos); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <button type="button" class="btn btn-sm btn-warning" 
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#modalSalida<?php echo $visita['id']; ?>">
                                                <i class="fas fa-sign-out-alt me-1"></i> Registrar Salida
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div class="mt-3">
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                <strong><?php echo count($visitas_activas); ?> visitas activas encontradas.</strong> 
                                Para registrar una salida, haga clic en el botón correspondiente.
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Modales para registrar salida -->
    <?php foreach ($visitas_activas as $visita): ?>
    <div class="modal fade" id="modalSalida<?php echo $visita['id']; ?>" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-sign-out-alt me-2"></i>Registrar Salida
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" action="">
                    <div class="modal-body">
                        <div class="mb-3">
                            <p><strong>Visitante:</strong> <?php echo htmlspecialchars($visita['visitante']); ?></p>
                            <p><strong>DNI:</strong> <?php echo $visita['documento_visitante']; ?></p>
                            <p><strong>Hora Ingreso:</strong> <?php echo date('H:i', strtotime($visita['hora_ingreso'])); ?></p>
                            <p><strong>Área:</strong> <?php echo htmlspecialchars($visita['area'] ?? 'No especificada'); ?></p>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label">Tipo de Registro</label>
                            <select class="form-select" name="tipo_salida" id="tipo_salida_<?php echo $visita['id']; ?>" required>
                                <option value="automatico">Automático (Hora actual)</option>
                                <option value="manual">Manual (Ingresar hora)</option>
                            </select>
                        </div>
                        
                        <div class="mb-3" id="hora_salida_container_<?php echo $visita['id']; ?>" style="display: none;">
                            <label class="form-label">Hora de Salida</label>
                            <input type="time" class="form-control" name="hora_salida" 
                                   value="<?php echo date('H:i'); ?>">
                        </div>
                        
                        <input type="hidden" name="visita_id" value="<?php echo $visita['id']; ?>">
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Confirmar Salida</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    <?php endforeach; ?>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Inicializar tooltips
        document.addEventListener('DOMContentLoaded', function() {
            var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
            
            // Configurar todos los modales
            <?php foreach ($visitas_activas as $visita): ?>
            var selectElement = document.getElementById('tipo_salida_<?php echo $visita['id']; ?>');
            var containerElement = document.getElementById('hora_salida_container_<?php echo $visita['id']; ?>');
            
            if (selectElement && containerElement) {
                selectElement.addEventListener('change', function() {
                    if (this.value === 'manual') {
                        containerElement.style.display = 'block';
                    } else {
                        containerElement.style.display = 'none';
                    }
                });
            }
            <?php endforeach; ?>
            
            // Actualizar tiempos en tiempo real
            function actualizarTiempos() {
                document.querySelectorAll('tbody tr').forEach(function(row) {
                    var badge = row.querySelector('.badge-tiempo');
                    if (badge) {
                        var tiempo = badge.textContent.trim().split(':');
                        var horas = parseInt(tiempo[0]);
                        var minutos = parseInt(tiempo[1]);
                        
                        // Incrementar minutos
                        minutos++;
                        if (minutos >= 60) {
                            minutos = 0;
                            horas++;
                        }
                        
                        // Actualizar badge
                        var nuevoTiempo = sprintf("%02d:%02d", horas, minutos);
                        badge.textContent = '⏱️ ' + nuevoTiempo;
                        
                        // Actualizar clase según horas
                        badge.classList.remove('bg-success', 'bg-warning', 'bg-danger');
                        if (horas >= 2) {
                            badge.classList.add('bg-danger');
                        } else if (horas >= 1) {
                            badge.classList.add('bg-warning');
                        } else {
                            badge.classList.add('bg-success');
                        }
                    }
                });
            }
            
            // Actualizar cada minuto
            setInterval(actualizarTiempos, 60000);
            
            // Función helper para formatear
            function sprintf(format, ...args) {
                return format.replace(/%(\d*)d/g, function(match, width) {
                    var num = args.shift();
                    width = width || 0;
                    return num.toString().padStart(width, '0');
                }).replace(/%(\d*)s/g, function(match, width) {
                    var str = args.shift();
                    width = width || 0;
                    return str.padStart(width, ' ');
                });
            }
        });
        
        // Auto-enfocar en el primer campo del modal cuando se abre
        document.addEventListener('shown.bs.modal', function(event) {
            var modal = event.target;
            var input = modal.querySelector('input, select');
            if (input) {
                input.focus();
            }
        });
    </script>
</body>
</html>