<?php
require_once 'config/database.php';

$database = new Database();
$db = $database->getConnection();

$mensaje = '';
$tipo_mensaje = '';
$token_valido = false;
$token = isset($_GET['token']) ? $_GET['token'] : '';

// Verificar token
if (!empty($token)) {
    $query = "SELECT id FROM usuarios WHERE token_recuperacion = :token AND activo = 1";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':token', $token);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $usuario = $stmt->fetch(PDO::FETCH_ASSOC);
        $token_valido = true;
        $user_id = $usuario['id'];
    } else {
        $mensaje = "El enlace de recuperación es inválido o ha expirado.";
        $tipo_mensaje = "danger";
    }
}

if ($_SERVER["REQUEST_METHOD"] == "POST" && $token_valido) {
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    
    // Validar contraseñas
    if ($password !== $confirm_password) {
        $mensaje = "Las contraseñas no coinciden.";
        $tipo_mensaje = "danger";
    } elseif (strlen($password) < 6) {
        $mensaje = "La contraseña debe tener al menos 6 caracteres.";
        $tipo_mensaje = "danger";
    } else {
        // Hash de la nueva contraseña
        $hashed_password = password_hash($password, PASSWORD_BCRYPT);
        
        // Actualizar contraseña y limpiar token
        $query = "UPDATE usuarios SET password = :password, token_recuperacion = NULL WHERE id = :id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':password', $hashed_password);
        $stmt->bindParam(':id', $user_id);
        
        if ($stmt->execute()) {
            $mensaje = "Contraseña actualizada exitosamente. Ahora puedes iniciar sesión con tu nueva contraseña.";
            $tipo_mensaje = "success";
            $token_valido = false; // Ya no mostrar el formulario
        } else {
            $mensaje = "Error al actualizar la contraseña. Por favor, intenta nuevamente.";
            $tipo_mensaje = "danger";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Restablecer Contraseña - Sistema de Visitas</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #e3f2fd 0%, #bbdefb 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
        }
        .reset-card {
            background: white;
            border-radius: 20px;
            box-shadow: 0 15px 35px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            animation: fadeIn 0.5s ease-out;
        }
        .reset-header {
            background: linear-gradient(to right, #28a745, #20c997);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .reset-body {
            padding: 30px;
        }
        .btn-reset {
            background: linear-gradient(to right, #28a745, #20c997);
            border: none;
            padding: 12px;
            font-weight: 600;
            transition: all 0.3s;
        }
        .btn-reset:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(40, 167, 69, 0.4);
        }
        .reset-icon {
            width: 80px;
            height: 80px;
            background: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 20px;
        }
        .password-strength {
            height: 5px;
            border-radius: 3px;
            margin-top: 5px;
            transition: all 0.3s;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-md-6 col-lg-5">
                <div class="reset-card">
                    <div class="reset-header">
                        <div class="reset-icon">
                            <i class="fas fa-lock fa-2x" style="color: #28a745;"></i>
                        </div>
                        <h3>Restablecer Contraseña</h3>
                        <p class="mb-0">Sistema de Visitas</p>
                    </div>
                    <div class="reset-body">
                        <?php if ($mensaje): ?>
                            <div class="alert alert-<?php echo $tipo_mensaje; ?> alert-dismissible fade show" role="alert">
                                <?php echo $mensaje; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($token_valido && empty($mensaje)): ?>
                            <p class="text-muted mb-4">
                                Ingresa tu nueva contraseña. Asegúrate de que sea segura y fácil de recordar.
                            </p>
                            
                            <form method="POST" action="" id="resetForm">
                                <div class="mb-3">
                                    <label for="password" class="form-label">Nueva Contraseña</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                        <input type="password" class="form-control" id="password" name="password" 
                                               required minlength="6" placeholder="Mínimo 6 caracteres">
                                        <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </div>
                                    <div class="password-strength" id="passwordStrength"></div>
                                    <small class="text-muted" id="passwordHint"></small>
                                </div>
                                
                                <div class="mb-4">
                                    <label for="confirm_password" class="form-label">Confirmar Contraseña</label>
                                    <div class="input-group">
                                        <span class="input-group-text"><i class="fas fa-lock"></i></span>
                                        <input type="password" class="form-control" id="confirm_password" 
                                               name="confirm_password" required minlength="6" placeholder="Repite la contraseña">
                                    </div>
                                    <small class="text-muted" id="passwordMatch"></small>
                                </div>
                                
                                <div class="d-grid mb-3">
                                    <button type="submit" class="btn btn-reset" id="submitBtn">
                                        <i class="fas fa-save me-2"></i>Actualizar Contraseña
                                    </button>
                                </div>
                            </form>
                        <?php elseif (empty($token) || !$token_valido): ?>
                            <div class="text-center">
                                <i class="fas fa-exclamation-triangle fa-3x text-warning mb-3"></i>
                                <p class="mb-3">El enlace de recuperación es inválido o ha expirado.</p>
                                <a href="recuperar.php" class="btn btn-primary">
                                    <i class="fas fa-redo me-1"></i>Solicitar Nuevo Enlace
                                </a>
                                <hr class="my-3">
                                <a href="login.php" class="text-decoration-none">
                                    <i class="fas fa-arrow-left me-1"></i>Volver al Inicio de Sesión
                                </a>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($tipo_mensaje == 'success'): ?>
                            <div class="text-center mt-3">
                                <a href="login.php" class="btn btn-primary">
                                    <i class="fas fa-sign-in-alt me-1"></i>Ir al Inicio de Sesión
                                </a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://kit.fontawesome.com/your-fontawesome-kit.js" crossorigin="anonymous"></script>
    <script>
        // Mostrar/ocultar contraseña
        document.getElementById('togglePassword').addEventListener('click', function() {
            const passwordInput = document.getElementById('password');
            const icon = this.querySelector('i');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                passwordInput.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        });
        
        // Validar fortaleza de contraseña
        document.getElementById('password').addEventListener('input', function() {
            const password = this.value;
            const strengthBar = document.getElementById('passwordStrength');
            const hint = document.getElementById('passwordHint');
            
            let strength = 0;
            let color = '#dc3545';
            let message = 'Muy débil';
            
            // Verificar longitud
            if (password.length >= 6) strength++;
            if (password.length >= 8) strength++;
            
            // Verificar caracteres diversos
            if (/[a-z]/.test(password)) strength++;
            if (/[A-Z]/.test(password)) strength++;
            if (/[0-9]/.test(password)) strength++;
            if (/[^A-Za-z0-9]/.test(password)) strength++;
            
            // Determinar color y mensaje
            switch (strength) {
                case 0:
                case 1:
                    color = '#dc3545';
                    message = 'Muy débil';
                    break;
                case 2:
                    color = '#fd7e14';
                    message = 'Débil';
                    break;
                case 3:
                    color = '#ffc107';
                    message = 'Regular';
                    break;
                case 4:
                    color = '#20c997';
                    message = 'Buena';
                    break;
                case 5:
                case 6:
                    color = '#28a745';
                    message = 'Excelente';
                    break;
            }
            
            // Actualizar barra y mensaje
            strengthBar.style.width = (strength / 6 * 100) + '%';
            strengthBar.style.backgroundColor = color;
            hint.textContent = message;
            hint.style.color = color;
        });
        
        // Verificar que las contraseñas coincidan
        document.getElementById('confirm_password').addEventListener('input', function() {
            const password = document.getElementById('password').value;
            const confirmPassword = this.value;
            const matchText = document.getElementById('passwordMatch');
            
            if (confirmPassword === '') {
                matchText.textContent = '';
                matchText.style.color = '';
            } else if (password === confirmPassword) {
                matchText.textContent = '✓ Las contraseñas coinciden';
                matchText.style.color = '#28a745';
            } else {
                matchText.textContent = '✗ Las contraseñas no coinciden';
                matchText.style.color = '#dc3545';
            }
        });
        
        // Validar formulario antes de enviar
        document.getElementById('resetForm').addEventListener('submit', function(e) {
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            if (password !== confirmPassword) {
                e.preventDefault();
                alert('Las contraseñas no coinciden. Por favor, verifica.');
                return false;
            }
            
            if (password.length < 6) {
                e.preventDefault();
                alert('La contraseña debe tener al menos 6 caracteres.');
                return false;
            }
            
            return true;
        });
    </script>
</body>
</html>